;****************************************************************************************
;* boot.s - Bootloader									*
;*											*
;* - clears the screen									*
;* - loads SETUP into memory (0000h:0500h)						*
;* - jumps to SETUP									*
;*											*
;* written by Bastian Gloeckle (MrSaint), programmer of nucleOS				*
;*											*
;* Copyright (C) 2004 by nucleOS group							*
;*											*
;* This program is free software; you can redistribute it and/or modify it under the	*
;* terms of the GNU General Public License as published by the Free Software Foundation	*
;* (version 2, June 1991)								*
;*											*
;* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*
;* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*
;* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*
;*											*
;* You should have received a copy of the GNU General Public License along with this	*
;* program; if not, write to: 								*
;* Free Software Foundation, Inc.							*
;* 59 Temple Place, Suite 330								*
;* Boston, MA 02111-1307 USA								*
;*											*
;*											*
;* You can contanct me by electronic mail: admin@saint-soft.de				*
;****************************************************************************************

ORG 0x7C00
BITS 16

;:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
; code entry
;:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
start:	
	mov ax, cs			; be sure that ds = cs
	mov ds, ax

	mov byte [bootdrive], dl	; save boot drive

	cli

	mov ax, 100h			; set up stack
	mov ss, ax
	mov sp, 400h			; stack = 1024 bytes long

	sti

	call clear_screen		; clear the screen
	
	mov si, booting			; output boot message
	call output_str_real
	mov si, new_line
	call output_str_real
	mov si, new_line
	call output_str_real

	mov si, loading			; output loading message
	call output_str_real
	mov si, new_line
	call output_str_real

	pusha

	call load_setup			; load SETUP to memory

	call shut_down_floppy_motor	; shut down floppy motor

	popa

	mov si, jumping			; output jumping message
	call output_str_real
	mov si, new_line
	call output_str_real
	mov si, new_line
	call output_str_real

	mov dl, byte [bootdrive]	; put bootdrive back to dl (because SETUP reads it once again)

	jmp 500h;			; jump to SETUP (near jump beacause they are in the same segment 0000h)

	mov ax, 01h			; output error if we get here (should never happen)
	call output_error

	jmp halt


;:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
; data and 16bit functions
;:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
booting:	db "Booting nucleOS...",0
loading:	db "Loading SETUP...",0
jumping:	db "Jumping to SETUP...",0
error:		db "ERROR: BOOT-", 0
new_line:	db 13,10,0
cursor_top:	db 0
bootdrive:	db 0


; ........................................................
; outputs an error message
;
; INPUT:
;	AX: error number
; ........................................................
output_error:
	push si				; first output the error message
	mov si, error
	call output_str_real
	call AXtoString
	mov ax, di
	mov si, ax
	call output_str_real		; output it
	pop si
	ret

; ........................................................
; outputs a string in real mode using BIOS Interrupt 10h
;
; DS:SI = Adress of source string
; string must be 0-terminated
; CLEARS THE DIRECTION FLAG!
; ........................................................
output_str_real:
	push ax				; save ax and bx
	push bx
	cld
	mov ah, 0Eh			; function 0Eh: teletype output
	mov bh, 00h			; first page
	.entry:
		lodsb			; load one byte from DS:SI
		or al, al
		jz .end
		int 10h			; output it
		jmp .entry
	.end:
		pop bx			; restore bx and ax
		pop ax
		ret

; ........................................................
; shuts down the floppy motor.
; ........................................................
shut_down_floppy_motor:
	push dx
	push ax
	mov dx, 03F2h
	in al, dx		; get status of port 03F2h
	and al, 0Fh		; disable all motor-on bits
	out dx, al		; send back to port 03F2h
	pop ax
	pop dx
	ret


; ........................................................
; loads SETUP from the bootdrive into memory at
; 0000h:0500h
; using BIOS-Interrupt 13h
; ........................................................
load_setup:
	push ds
	mov ax, 0		; reset the drive
	mov ax, 0
	mov dl, byte [bootdrive]; this drive
	int 13h			; do it :) (BIOS Interrupt)
	pop ds
	jc load_setup		; If carry flag is set, something went wrong -> try again (-> maybe endless loop)

	.read:
		mov ax, 0000h		; put destination adress in ES:BX (0000h:0500h)
		mov es, ax
		mov bx, 0500h
		mov ah, 2		; read sectors from drive
		mov al, 7		; read 7 sectors (one sector = 512 Byte)
		mov ch, 0		; use cylinder 0
		mov cl, 2		; start at sector 2 (because in Sector 1 is the bootloader)
		mov dh, 0		; head=0
		mov dl, byte [bootdrive]
		int 13h			; read the sectors to ES:BX
		jc .read		; If carry -> try again

	.read2:
		mov ax, 0000h		; put destination adress in ES:BX (0000h:0500h)
		mov es, ax
		mov bx, 1300h
		mov ah, 1		; read sectors from drive
		mov al, 1		; read 7 sectors (one sector = 512 Byte)
		mov ch, 1		; use cylinder 0
		mov cl, 2		; start at sector 2 (because in Sector 1 is the bootloader)
		mov dh, 0		; head=0
		mov dl, byte [bootdrive]
		int 13h			; read the sectors to ES:BX
		jc .read2		; If carry -> try again
	ret

; ........................................................
; clears the screen and sets the cursor to the top left 
; corner
; uses BIOS Interrupt 10h
; ........................................................
clear_screen:
	push di
	push ax
	push fs
	mov ax, 0B800h			; segment of video buffer
	mov fs, ax
	xor ax, ax
	jmp .entry_first		; dont increase ax the first time
	.entry:
		inc ax
	.entry_first:			; clear the screen by dirctly accessing the memory
		mov di, ax
		mov [fs:di], byte 00h	; write character #0 to the memory
		inc ax
		mov di, ax
		mov [fs:di], byte 07h	; attribute
		cmp ax, 2*80*25-1	; 25 lines; 80 characters per line
		jne .entry
	mov ah, 02h			; set cursor position
	mov bh, 00h
	xor dx, dx			; position: 0h x 0h (top left corner)
	int 10h
	pop fs
	pop ax
	pop di
	ret

; ........................................................
; converts the content of AX to a readable string in hex
; format
;
; INPUT:
;	AX	number to convert
;
; OUTPUT:
;	DS:DI	Adress of AXtoString_store
; ........................................................
AXtoString_store:	db 0,0,0,0,'h',0	; Output buffer
AXtoString:
	push bx
	mov bx, ax
	and bx, 0F0F0h		; get the 4 higher bits of BH and BL
	shr bx, 4		; 4 bitshifts right (move the filtered higher bits to the lower positions)
	add bx, 3030h		; add 48 (ASCII 48 = "0")
	cmp bh, 57
	jng .DoNotAdd_1
	add bx, 0700h		; if > ASCII 57 ("9") => add 7 (-> ASCII 65 = "A")
	.DoNotAdd_1:
		cmp bl, 57
		jng .store_1
		add bx, 0007h		; if > ASCII 57 ("9") => add 7 (-> ASCII 65 = "A")
	.store_1:
		mov [AXtoString_store], byte bh
		mov [AXtoString_store+2], byte bl
		mov bx, ax
		and bx, 00F0Fh		; [see above]
		add bx, 3030h
		cmp bh, 57
		jng .DoNotAdd_2
		add bx, 0700h
	.DoNotAdd_2:
		cmp bl, 57
		jng .store_2
		add bx, 0007h
	.store_2:
		mov [AXtoString_store+1], byte bh
		mov [AXtoString_store+3], byte bl
	
	pop bx
	lea di, [AXtoString_store]
	ret

; ........................................................
; halts the CPU
; ........................................................
halt:
	cli
	hlt


times ((512-($-$$)-2)/2)-1 dw 0FAF4h		; print many "cli" and "hlt"s :) (until we are at byte 509)
times 512-($-$$)-2 db 0F4h			; so we have "hlt" at the end (byte 510 is hlt)
dw 0AA55h					; Bootsignature (byte 511 and 512)